<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Post\Comment\Like;

use Erlage\Photogram\System;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Data\Models\Post\PostModel;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Tables\Post\PostCommentLikeTable;
use Erlage\Photogram\Data\Dtos\User\UserMetaPushSettingsDTO;
use Erlage\Photogram\Data\Models\Notification\NotificationEnum;
use Erlage\Photogram\Data\Models\Post\Comment\PostCommentModel;
use Erlage\Photogram\Data\Tables\Notification\NotificationTable;
use Erlage\Photogram\Data\Models\Notification\NotificationFinder;
use Erlage\Photogram\Data\Models\Notification\NotificationBuilder;
use Erlage\Photogram\Data\Models\Post\Comment\PostCommentModelHelper;
use Erlage\Photogram\Data\Dtos\Notification\NotificationLinkedContentDTO;
use Erlage\Photogram\Data\Models\Post\Comment\Like\PostCommentLikeFinder;
use Erlage\Photogram\Data\Models\Post\Comment\Like\PostCommentLikeBuilder;

final class PostCommentLikeActions extends ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | do like
    |--------------------------------------------------------------------------
    */

    public static function add(): void
    {
        self::postCommentLikeProcessor(true);
    }

    /*
    |--------------------------------------------------------------------------
    | undo like
    |--------------------------------------------------------------------------
    */

    public static function remove(): void
    {
        self::postCommentLikeProcessor(false);
    }

    /*
    |--------------------------------------------------------------------------
    | processor
    |--------------------------------------------------------------------------
    */

    private static function postCommentLikeProcessor(bool $doLike): void
    {
        self::process(function () use ($doLike)
        {
            /*
            |--------------------------------------------------------------------------
            | flags
            |--------------------------------------------------------------------------
            */

            $flagUpdateCache = false;

            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $postCommentIdFromReq = self::$request -> findKey(
                PostCommentLikeTable::LIKED_POST_COMMENT_ID,
                RequestTable::PAYLOAD,
                PostCommentLikeTable::TABLE_NAME
            );

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $postCommentIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure comment exists
            |--------------------------------------------------------------------------
            */

            $targetPostCommentModel = PostCommentModel::findFromId_throwException($postCommentIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure post exists
            |--------------------------------------------------------------------------
            */

            $targetPostModel = PostModel::findFromId_throwException($targetPostCommentModel -> getParentPostId());

            /*
            |--------------------------------------------------------------------------
            | ensure post owner exists
            |--------------------------------------------------------------------------
            */

            $postOwnerUserModel = UserModel::findFromId_throwException($targetPostModel -> getOwnerUserId());

            /*
            |--------------------------------------------------------------------------
            | privacy checks
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserContentAvailable($postOwnerUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | if undo like request
            |--------------------------------------------------------------------------
            */

            if (false == $doLike)
            {
                $finder = (new PostCommentLikeFinder())
                    -> setLikedPostCommentId($targetPostCommentModel -> getId())
                    -> setLikedByUserId(self::$authedUserModel -> getId())
                    -> find();

                if ($finder -> isFound())
                {
                    $finder -> popModelFromResults() -> delete();

                    $flagUpdateCache = true;
                }
            }
            /*
            |--------------------------------------------------------------------------
            | if it's a do like request
            |--------------------------------------------------------------------------
            */
            elseif (true == $doLike)
            {
                /*
                |--------------------------------------------------------------------------
                | try selecting the post like first
                |--------------------------------------------------------------------------
                */

                $postCommentLikeBean = (new Query())
                    -> from(PostCommentLikeTable::TABLE_NAME)
                    -> where(PostCommentLikeTable::LIKED_BY_USER_ID, self::$authedUserModel -> getId())
                    -> where(PostCommentLikeTable::PARENT_POST_ID, $targetPostModel -> getId())
                    -> where(PostCommentLikeTable::LIKED_POST_COMMENT_ID, $targetPostCommentModel -> getId())
                    -> selectOne();

                /*
                |--------------------------------------------------------------------------
                | if is already liked
                |--------------------------------------------------------------------------
                */

                if (null != $postCommentLikeBean)
                {
                    /*
                    |--------------------------------------------------------------------------
                    | create model from fetched bean
                    |--------------------------------------------------------------------------
                    */

                    $postCommentLikeModel = PostCommentModel::createFromUntouchedBean_noException($postCommentLikeBean);

                    self::ensureModel($postCommentLikeModel);

                    /*
                    |--------------------------------------------------------------------------
                    | add to model to response
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(PostCommentLikeTable::getTableName(), $postCommentLikeModel -> getDataMap());
                }

                /*
                |--------------------------------------------------------------------------
                | else if is not already liked
                |--------------------------------------------------------------------------
                */

                else
                {
                    /*
                    |--------------------------------------------------------------------------
                    | build like
                    |--------------------------------------------------------------------------
                    */

                    $postCommentLikeModel = (new PostCommentLikeBuilder())
                        -> setLikedByUserId(self::$authedUserModel -> getId())
                        -> setParentPostId($targetPostModel -> getId())
                        -> setLikedPostCommentId($targetPostCommentModel -> getId())
                        -> dispense();

                    /*
                    |--------------------------------------------------------------------------
                    | do like
                    |--------------------------------------------------------------------------
                    */

                    $postCommentLikeModel -> save();

                    /*
                    |--------------------------------------------------------------------------
                    |  if everything is alright, add to map
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(PostCommentLikeTable::getTableName(), $postCommentLikeModel -> getDataMap());

                    $flagUpdateCache = true;
                }

                /*
                |--------------------------------------------------------------------------
                |  add current user, either way
                |--------------------------------------------------------------------------
                */
                self::addToResponse(UserTable::getTableName(), self::$authedUserModel -> getDataMap());
            }

            if ($flagUpdateCache)
            {
                PostCommentModelHelper::updateCacheLikesCount($targetPostCommentModel);

                /*
                |--------------------------------------------------------------------------
                | add notification
                |--------------------------------------------------------------------------
                */

                if ($doLike)
                {
                    $notifyCommentOwner = false;

                    /*
                    |--------------------------------------------------------------------------
                    | check whether comment owner has opted in for notifications
                    |--------------------------------------------------------------------------
                    */

                    if ($targetPostCommentModel -> getOwnerUserId() != self::$authedUserModel -> getId())
                    {
                        $targetPostCommentOwnerModel = UserModel::findFromId_noException(
                            $targetPostCommentModel -> getOwnerUserId()
                        );

                        if ($targetPostCommentOwnerModel -> isModel())
                        {
                            $notifyCommentOwner = $targetPostCommentOwnerModel
                                -> getMetaPushSettings()
                                -> getCommentLikesAndPins() == UserMetaPushSettingsDTO::VALUE_ON;
                        }
                    }

                    if ($notifyCommentOwner)
                    {
                        /*
                        |--------------------------------------------------------------------------
                        | first try to find if someone already liked post today
                        |--------------------------------------------------------------------------
                        */

                        $finder = (new NotificationFinder())
                            -> setToUserId($targetPostCommentOwnerModel -> getId())
                            -> setTargetContentId($targetPostCommentModel -> getId())
                            -> setMetaType(NotificationEnum::META_TYPE_POST_COMMENT_LIKE)
                            -> setStampRegistrationDate(System::isoDate())
                            -> find();

                        if ($finder -> isFound())
                        {
                            /*
                            |--------------------------------------------------------------------------
                            | notification exists, simply add user to existing notifcation
                            |--------------------------------------------------------------------------
                            */

                            $notifcationModel = $finder -> popModelFromResults();

                            $notificationContent = $notifcationModel -> getLinkedContent();

                            if ( ! $notificationContent -> getUserIds() -> containsValue(self::$authedUserModel -> getId()))
                            {
                                $notificationContent -> linkUserId(self::$authedUserModel -> getId());

                                $notifcationModel -> update(
                                    array(
                                        NotificationTable::LINKED_CONTENT => $notificationContent,
                                        NotificationTable::META_IS_READ   => NotificationEnum::META_IS_READ_NO,
                                    )
                                );

                                $notifcationModel -> save();
                            }
                        }
                        else
                        {
                            /*
                            |--------------------------------------------------------------------------
                            | notification doesnt exists, create and dispatch a new notification
                            |--------------------------------------------------------------------------
                            */

                            $notificationContent = new NotificationLinkedContentDTO();

                            $notificationContent -> linkUserId(self::$authedUserModel -> getId());
                            $notificationContent -> linkPostId($targetPostModel -> getId());

                            $notificationModel = (new NotificationBuilder())
                                -> setToUserId($targetPostCommentOwnerModel -> getId())
                                -> setTargetContentId($targetPostCommentModel -> getId())
                                -> setMetaType(NotificationEnum::META_TYPE_POST_COMMENT_LIKE)
                                -> setLinkedContent($notificationContent)
                                -> dispense();

                            $notificationModel -> save();
                        }
                    }
                }
            }
        });
    }
}
